import React, { useState, useEffect, useCallback, memo } from 'react';

const Flowmeter = memo(({ totalFlow = 0 }) => {
  // Use integer state to minimize re-renders
  const [bobbinPosition, setBobbinPosition] = useState(95);
  
  // Memoize position calculation for performance
  const calculateBobbinPosition = useCallback((flow) => {
    if (flow < 1) {
      return Math.min(95, Math.max(5, 88 - 14 * flow));
    } else if (flow < 2) {
      return Math.min(74, Math.max(5, 74 - 16 * (flow - 1)));
    } else if (flow < 5) {
      return Math.min(58, Math.max(5, 58 - 6 * (flow - 2)));
    } else if (flow < 10) {
      return Math.min(41, Math.max(5, 41 - 3 * (flow - 5)));
    } else if (flow < 20) {
      return Math.min(25, Math.max(5, 25 - 1.5 * (flow - 10)));
    } else {
      return Math.max(5, 10 - (flow - 20));
    }
  }, []);

  useEffect(() => {
    // Batch updates for smoother animation
    requestAnimationFrame(() => {
      setBobbinPosition(calculateBobbinPosition(totalFlow));
    });
  }, [totalFlow, calculateBobbinPosition]);

  // Constants for efficient re-renders
  const numberMarks = [4, 3, 2, 1];
  const decimalMarks = [0.8, 0.6, 0.4, 0.2];
  const lineOffsets = [-13, 19, 51, 82];

  return (
    <div className="flex flex-col items-center w-28 bg-gray-100 p-2 rounded-lg">
      {/* Cylinder top */}
      <div className="w-full h-6 bg-gray-300 rounded-t-full" />
      
      {/* Main tube */}
      <div className="relative w-24 h-96 bg-gradient-to-r from-gray-300 via-gray-200 to-gray-300 my-2 rounded-lg overflow-hidden">
        {/* Inner tube gradient */}
        <div className="absolute inset-x-2 inset-y-1 bg-gradient-to-r from-white/30 via-white/90 to-white/10 rounded-full">
          {/* Bobbin */}
          <div className="absolute left-1/2 w-4 h-4 bg-green-500 rounded-full -translate-x-1/2 transition-all duration-300 ease-out" 
               style={{ top: `${bobbinPosition}%` }} />
        </div>
        
        {/* Scale markings */}
        <div className="absolute inset-x-0 h-full flex flex-col py-4">
          <div className="h-8" />
          
          {/* Integer markings */}
          <div className="flex-grow flex flex-col justify-between relative">
            {numberMarks.map((value, index) => (
              <React.Fragment key={value}>
                <div className="flex items-center justify-between px-4">
                  <div className="w-3 h-px bg-black" />
                  <span className="text-xs">{value}</span>
                  <div className="w-3 h-px bg-black" />
                </div>
                <div className="absolute left-1/2 w-5 h-px bg-black -translate-x-1/2" 
                     style={{ top: `${lineOffsets[index]}%` }} />
              </React.Fragment>
            ))}
          </div>
          
          {/* Decimal markings */}
          <div className="flex flex-col space-y-1 mt-2">
            {decimalMarks.map(value => (
              <div key={value} className="flex items-center justify-between px-4">
                <div className="w-3 h-px bg-black" />
                <span className="text-xs">{value.toFixed(1)}</span>
                <div className="w-3 h-px bg-black" />
              </div>
            ))}
          </div>
        </div>
      </div>
      
      {/* Cylinder bottom */}
      <div className="w-full h-6 bg-gray-300 rounded-b-full" />
      
      {/* Control knob */}
      <div className="relative w-20 h-20 mt-4">
        <div className="absolute inset-0 bg-gradient-to-br from-gray-300 via-gray-400 to-gray-600 rounded-full">
          <div className="absolute inset-2 bg-gradient-to-br from-gray-200 via-gray-300 to-gray-500 rounded-full flex items-center justify-center">
            <div className="w-1 h-10 bg-gray-700 rounded-full" />
          </div>
        </div>
      </div>
    </div>
  );
});

Flowmeter.displayName = 'Flowmeter';

export default Flowmeter;