<?php
// File: /includes/session_manager.php

// Load session settings from database
function getSessionExpirySeconds() {
    global $pdo;
    
    $defaults = ['session_expiry_amount' => 24, 'session_expiry_unit' => 'hours'];
    
    try {
        if (!isset($pdo)) {
            $pdo = new PDO("mysql:host=" . DB_HOST . ";dbname=" . DB_NAME, DB_USER, DB_PASS);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        }
        
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM wp_settings WHERE setting_key IN ('session_expiry_amount', 'session_expiry_unit')");
        $stmt->execute();
        $settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
        
        if (!empty($settings)) {
            $defaults = array_merge($defaults, $settings);
        }
    } catch (Exception $e) {
        error_log("Session settings DB error: " . $e->getMessage());
    }
    
    $amount = (int)$defaults['session_expiry_amount'];
    $unit = $defaults['session_expiry_unit'];
    
    switch($unit) {
        case 'seconds':
            return $amount;
        case 'minutes':
            return $amount * 60;
        case 'hours':
            return $amount * 3600;
        case 'days':
            return $amount * 86400;
        case 'months':
            return $amount * 2592000; // 30 days
        default:
            return 86400; // 24 hours default
    }
}

// Start session with configurable expiry
if (session_status() == PHP_SESSION_NONE) {
    $session_timeout = getSessionExpirySeconds();
    session_set_cookie_params($session_timeout);
    session_start();
    
    // Set expiry for authenticated pages
    $current_page = basename($_SERVER['PHP_SELF']);
    if ($current_page !== 'login.php' && $current_page !== 'register.php' && $current_page !== 'index.php') {
        $_SESSION['SESSION_EXPIRES'] = time() + $session_timeout;
    } else {
        // Force refresh existing sessions with current timeout settings
        $current_page = basename($_SERVER['PHP_SELF']);
        if ($current_page !== 'login.php' && $current_page !== 'register.php' && $current_page !== 'index.php') {
            $_SESSION['SESSION_EXPIRES'] = time() + $session_timeout;
        }
    }
} else {
    // Session already started - check if we need to update expiry with new settings
    $session_timeout = getSessionExpirySeconds();
    $current_page = basename($_SERVER['PHP_SELF']);
    
    if ($current_page !== 'login.php' && $current_page !== 'register.php' && $current_page !== 'index.php') {
        // Always refresh session expiry to match current database settings
        $_SESSION['SESSION_EXPIRES'] = time() + $session_timeout;
    }
}

// Check session and license validity
$current_page = basename($_SERVER['PHP_SELF']);
$is_login_page = ($current_page === 'login.php' || $current_page === 'register.php' || $current_page === 'user_a.php');

if (!$is_login_page) {
    if (isset($_SESSION['user_id']) || isset($_SESSION['front_user_id'])) {
        $current_time = time();
        $userId = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : $_SESSION['front_user_id'];
        
        // Check session expiry
        if (isset($_SESSION['SESSION_EXPIRES']) && $current_time > $_SESSION['SESSION_EXPIRES']) {
            $_SESSION['REDIRECT_URL'] = $_SERVER['REQUEST_URI'];
            session_unset();
            session_destroy();
            // Always return JSON for expired sessions - let JavaScript handle the popup
            if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] == 'XMLHttpRequest') {
                header('Content-Type: application/json');
                echo json_encode(['valid' => false, 'expired' => true, 'reason' => 'session_expired']);
                exit();
            }
            // For non-AJAX requests, session is expired but don't redirect
            // Let the 5-second check handle it with popup
        }
        
        // Check license expiration daily
        if (!isset($_SESSION['LAST_EXPIRATION_CHECK']) || ($current_time - $_SESSION['LAST_EXPIRATION_CHECK'] > 86400)) {
            if (!isset($pdo)) {
                $pdo = new PDO("mysql:host=" . DB_HOST . ";dbname=" . DB_NAME, DB_USER, DB_PASS);
                $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            }
            
            $stmt = $pdo->prepare("SELECT g.expiration_date FROM " . PREFIX . "users u LEFT JOIN " . PREFIX . "groups g ON u.group_id = g.id WHERE u.ID = ?");
            $stmt->execute([$userId]);
            $license = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($license && $license['expiration_date'] && strtotime($license['expiration_date']) < $current_time) {
                $_SESSION['REDIRECT_URL'] = $_SERVER['REQUEST_URI'];
                session_unset();
                session_destroy();
                if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] == 'XMLHttpRequest') {
                    header('Content-Type: application/json');
                    echo json_encode(['valid' => false, 'expired' => true, 'reason' => 'license_expired']);
                    exit();
                } else {
                    header("Location: /login.php?expired=1");
                    exit();
                }
            }
            
            $_SESSION['LAST_EXPIRATION_CHECK'] = $current_time;
        }
    }
}

// Function to refresh session expiry on successful reauth
function refreshSessionExpiry() {
    if (isset($_SESSION['user_id']) || isset($_SESSION['front_user_id'])) {
        $session_timeout = getSessionExpirySeconds();
        $_SESSION['SESSION_EXPIRES'] = time() + $session_timeout;
    }
}
?>