import os
from pathlib import Path
import argparse

def generate_directory_listing(start_path='.', output_file='directory_listing.txt', exclude_dirs=None, exclude_extensions=None):
    """
    Generate a formatted directory listing with relative paths.
    
    Args:
        start_path (str): Starting directory path
        output_file (str): Output file name
        exclude_dirs (list): List of directory names to exclude
        exclude_extensions (list): List of file extensions to exclude
    """
    if exclude_dirs is None:
        exclude_dirs = ['.git', 'node_modules', '__pycache__', 'venv']
    if exclude_extensions is None:
        exclude_extensions = ['.pyc', '.pyo', '.pyd']
    
    # Convert start_path to absolute path
    start_path = os.path.abspath(start_path)
    
    with open(output_file, 'w', encoding='utf-8') as f:
        f.write(f"Directory listing for: {start_path}\n")
        f.write("=" * 80 + "\n\n")
        
        for root, dirs, files in os.walk(start_path):
            # Remove excluded directories
            dirs[:] = [d for d in dirs if d not in exclude_dirs]
            
            # Get relative path
            rel_path = os.path.relpath(root, start_path)
            if rel_path == '.':
                rel_path = ''
            
            # Write directory name
            if rel_path:
                f.write(f"\n📁 {rel_path}/\n")
            
            # Write files
            for file in sorted(files):
                if not any(file.endswith(ext) for ext in exclude_extensions):
                    file_path = os.path.join(rel_path, file) if rel_path else file
                    f.write(f"    📄 {file_path}\n")

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description='Generate a directory listing')
    parser.add_argument('--path', default='.', help='Starting directory path')
    parser.add_argument('--output', default='directory_listing.txt', help='Output file name')
    parser.add_argument('--exclude-dirs', nargs='+', help='Directories to exclude')
    parser.add_argument('--exclude-ext', nargs='+', help='File extensions to exclude')
    
    args = parser.parse_args()
    
    generate_directory_listing(
        args.path,
        args.output,
        args.exclude_dirs,
        args.exclude_ext
    )