document.addEventListener('DOMContentLoaded', () => {
    // Constants
    const MAX_CELL_HEIGHT = 50; // maximum height in pixels
    const html = document.documentElement;
    const compactToggle = document.getElementById('compact-toggle');
    let expandedRow = null; // Keep track of the currently expanded row

    // Cookie functions
    function setCookie(name, value, days) {
        let expires = "";
        if (days) {
            const date = new Date();
            date.setTime(date.getTime() + (days * 24 * 60 * 60 * 1000));
            expires = "; expires=" + date.toUTCString();
        }
        document.cookie = name + "=" + (value || "") + expires + "; path=/";
    }

    function getCookie(name) {
        const nameEQ = name + "=";
        const ca = document.cookie.split(';');
        for(let i = 0; i < ca.length; i++) {
            let c = ca[i];
            while (c.charAt(0) === ' ') c = c.substring(1, c.length);
            if (c.indexOf(nameEQ) === 0) return c.substring(nameEQ.length, c.length);
        }
        return null;
    }

    // Function to make a single table cell scrollable
    function makeCellScrollable(cell) {
        if (!cell.querySelector('.cell-content-wrapper')) {
            const wrapper = document.createElement('div');
            wrapper.className = 'cell-content-wrapper';
            wrapper.style.maxHeight = MAX_CELL_HEIGHT + 'px';
            wrapper.style.overflowX = 'hidden';
            wrapper.style.overflowY = 'auto';
            wrapper.style.display = 'block';
            
            // Move the cell's content into the wrapper
            wrapper.innerHTML = cell.innerHTML;
            cell.innerHTML = '';
            cell.appendChild(wrapper);
        }
    }

    // Function to make all table cells scrollable
    function makeAllTableCellsScrollable() {
        const tableCells = document.querySelectorAll('td');
        tableCells.forEach(makeCellScrollable);
    }

    // Function to toggle compact mode
    function toggleCompactMode(isCompact) {
        const tableRows = document.querySelectorAll('tr');
        
        tableRows.forEach(row => {
            const cells = row.querySelectorAll('td');
            cells.forEach(cell => {
                makeCellScrollable(cell); // Ensure the cell is prepared
                const wrapper = cell.querySelector('.cell-content-wrapper');
                if (wrapper) {
                    if (isCompact && row !== expandedRow) {
                        wrapper.style.maxHeight = MAX_CELL_HEIGHT + 'px';
                        wrapper.style.overflowX = 'hidden';
                        wrapper.style.overflowY = 'auto';
                    } else {
                        wrapper.style.maxHeight = 'none';
                        wrapper.style.overflowY = 'visible';
                    }
                }
            });
        });
    }

    // Function to expand a row
    function expandRow(row) {
        if (expandedRow) {
            // Collapse the previously expanded row
            const cells = expandedRow.querySelectorAll('td');
            cells.forEach(cell => {
                const wrapper = cell.querySelector('.cell-content-wrapper');
                if (wrapper) {
                    wrapper.style.maxHeight = MAX_CELL_HEIGHT + 'px';
                    wrapper.style.overflowY = 'auto';
                }
            });
        }

        // Expand the clicked row
        const cells = row.querySelectorAll('td');
        cells.forEach(cell => {
            const wrapper = cell.querySelector('.cell-content-wrapper');
            if (wrapper) {
                wrapper.style.maxHeight = 'none';
                wrapper.style.overflowY = 'visible';
            }
        });

        expandedRow = row;
    }

    // Function to collapse all rows
    function collapseAllRows() {
        if (expandedRow) {
            const cells = expandedRow.querySelectorAll('td');
            cells.forEach(cell => {
                const wrapper = cell.querySelector('.cell-content-wrapper');
                if (wrapper) {
                    wrapper.style.maxHeight = MAX_CELL_HEIGHT + 'px';
                    wrapper.style.overflowY = 'auto';
                }
            });
            expandedRow = null;
        }
    }

    // Add event listeners
    document.addEventListener('click', (event) => {
        const isCompact = html.classList.contains('compact-mode');
        if (!isCompact) return; // Only handle clicks in compact mode

        const cell = event.target.closest('td');
        if (cell) {
            const row = cell.parentElement;
            if (row !== expandedRow) {
                expandRow(row);
            } else {
                collapseAllRows();
            }
        } else {
            collapseAllRows();
        }
    });

	// Compact toggle event listener
	compactToggle.addEventListener('click', () => {
		const willBeCompact = !html.classList.contains('compact-mode');
		html.classList.toggle('compact-mode');
		const icon = compactToggle.querySelector('i');
		const text = compactToggle.querySelector('span');

		if (willBeCompact) {
			icon.classList.replace('fa-expand', 'fa-compress');
			text.textContent = 'Compact Mode'; // swap these 4 to swap state
		} else {
			icon.classList.replace('fa-compress', 'fa-expand');
			text.textContent = 'Expanded Mode'; // swap these 4 to swap state
		}

		toggleCompactMode(willBeCompact); 
		collapseAllRows();
		
		setCookie('layout', willBeCompact ? 'compact' : 'normal', 30);
	});

    // Initialize on page load
    makeAllTableCellsScrollable();

    // Set compact mode as default
    const savedLayout = getCookie('layout') || 'compact';
    
    // Initialize compact mode by default
    html.classList.add('compact-mode');
    toggleCompactMode(true);

    // Set initial button state
    const icon = compactToggle.querySelector('i');
    const text = compactToggle.querySelector('span');
    
	if (savedLayout === 'normal') {
		// If user previously selected normal mode
		html.classList.remove('compact-mode');
		icon.classList.replace('fa-expand', 'fa-compress');
		text.textContent = 'Expanded Mode'; // swap these 4 to swap state
		toggleCompactMode(false);
	} else {
		// Default compact mode
		icon.classList.replace('fa-expand', 'fa-compress');
		text.textContent = 'Compact Mode'; // swap these 4 to swap state
	}

    // console.log('Saved Layout State:', savedLayout);
});

window.updateCompactMode = function() {
    const isCompact = document.documentElement.classList.contains('compact-mode');
    const MAX_CELL_HEIGHT = 50;
    
    function makeCellScrollable(cell) {
        if (!cell.querySelector('.cell-content-wrapper')) {
            const wrapper = document.createElement('div');
            wrapper.className = 'cell-content-wrapper';
            wrapper.style.maxHeight = isCompact ? MAX_CELL_HEIGHT + 'px' : 'none';
            wrapper.style.overflowX = isCompact ? 'hidden' : 'visible';
            wrapper.style.overflowY = isCompact ? 'auto' : 'visible';
            wrapper.style.display = 'block';
            wrapper.innerHTML = cell.innerHTML;
            cell.innerHTML = '';
            cell.appendChild(wrapper);
        }
    }
    
    document.querySelectorAll('td').forEach(cell => {
        makeCellScrollable(cell);
        const wrapper = cell.querySelector('.cell-content-wrapper');
        if (wrapper) {
            if (isCompact) {
                wrapper.style.maxHeight = MAX_CELL_HEIGHT + 'px';
                wrapper.style.overflowX = 'hidden';
                wrapper.style.overflowY = 'auto';
            } else {
                wrapper.style.maxHeight = 'none';
                wrapper.style.overflowY = 'visible';
            }
        }
    });
};