import os
import re
import json
import ast
from collections import OrderedDict

def js_to_python(js_object):
    js_object = js_object.replace('true', 'True').replace('false', 'False')
    js_object = js_object.replace('null', 'None')
    return ast.literal_eval(js_object)

def process_single_file(filename):
    js_dir = '/home/anesoft/public_html/dev.anesoft.com/app/cases_updated/'
    json_dir = '/home/anesoft/public_html/dev.anesoft.com/app/case_data/'
    
    if not os.path.exists(json_dir):
        os.makedirs(json_dir)
    
    print(f"\n--- Processing file: {filename} ---")
    with open(os.path.join(js_dir, filename), 'r') as f:
        content = f.read()

    pattern = r'CaseData\[(\d+)\]\s*=\s*({[\s\S]*?})(?:\s*;|\s*$)'
    match = re.search(pattern, content, re.DOTALL)
    
    if match:
        case_id = match.group(1)
        case_data = match.group(2)
        
        try:
            case_dict = js_to_python(case_data)

            new_case_dict = OrderedDict()
            for key, value in case_dict.items():
                new_case_dict[key] = value
                if key == "title":
                    new_case_dict["id"] = int(case_id)
                    new_case_dict["isPro"] = int(case_id) > 40

            # Extract the case number from the filename
            filename_case_number = re.search(r'Anesth6Case(\d+)\.js', filename).group(1)
            json_filename = f'case_{filename_case_number}.json'
            json_path = os.path.join(json_dir, json_filename)
            with open(json_path, 'w') as f:
                json.dump(new_case_dict, f, indent=2)
            
            print(f"Successfully converted {filename} to {json_filename}")
            return True, int(case_id), int(filename_case_number)
        except (SyntaxError, ValueError) as e:
            print(f"Error processing {filename}: {e}")
            return False, None, None
    else:
        print(f"No matching CaseData object found in: {filename}")
        return False, None, None

if __name__ == "__main__":
    js_dir = '/home/anesoft/public_html/app.anesoft.com/app/cases_updated/'
    processed_files = []
    unprocessed_files = []
    case_numbers = []
    filename_numbers = []
    
    # Get all files and sort them numerically by case number in filename
    all_files = [f for f in os.listdir(js_dir) if f.startswith('Anesth6Case') and f.endswith('.js')]
    all_files.sort(key=lambda x: int(re.search(r'Anesth6Case(\d+)\.js', x).group(1)))
    
    for filename in all_files:
        try:
            success, internal_case_number, filename_case_number = process_single_file(filename)
            if success:
                processed_files.append(filename)
                if internal_case_number:
                    case_numbers.append(internal_case_number)
                if filename_case_number:
                    filename_numbers.append(filename_case_number)
            else:
                unprocessed_files.append(filename)
        except Exception as e:
            print(f"Unexpected error processing {filename}: {e}")
            unprocessed_files.append(filename)
        
        # Print progress every 10 files
        if len(processed_files) % 10 == 0:
            print(f"Progress: Processed {len(processed_files)} files")
    
    print("\n--- Summary ---")
    print(f"Successfully processed files: {len(processed_files)}")
    print(f"Files that couldn't be processed: {len(unprocessed_files)}")
    
    print("\nProcessed files:")
    for file in processed_files:
        print(file)
    
    print("\nUnprocessed files:")
    for file in unprocessed_files:
        print(file)

    print("\nLast processed file:")
    if processed_files:
        print(processed_files[-1])
    else:
        print("No files were processed successfully.")

    print("\nTotal number of .js files in directory:", len(all_files))
    print("First file:", all_files[0])
    print("Last file:", all_files[-1])

    print("\nInternal case numbers processed:")
    print(f"Min: {min(case_numbers) if case_numbers else 'N/A'}")
    print(f"Max: {max(case_numbers) if case_numbers else 'N/A'}")
    print(f"Total unique case numbers: {len(set(case_numbers))}")
    print(f"Missing internal case numbers: {set(range(1, max(case_numbers) + 1)) - set(case_numbers) if case_numbers else 'N/A'}")

    print("\nFilename case numbers processed:")
    print(f"Min: {min(filename_numbers) if filename_numbers else 'N/A'}")
    print(f"Max: {max(filename_numbers) if filename_numbers else 'N/A'}")
    print(f"Total unique filename numbers: {len(set(filename_numbers))}")
    print(f"Missing filename numbers: {set(range(1, max(filename_numbers) + 1)) - set(filename_numbers) if filename_numbers else 'N/A'}")